/*
 * Copyright 2020 The Matrix.org Foundation C.I.C.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.matrix.android.sdk.api.session.room.model.call

import com.squareup.moshi.Json
import com.squareup.moshi.JsonClass

/**
 * This event is sent to signal the intent of a participant in a call to replace the call with another,
 * such that the other participant ends up in a call with a new user.
 */
@JsonClass(generateAdapter = true)
data class CallReplacesContent(
        /**
         * Required. The ID of the call this event relates to.
         */
        @Json(name = "call_id") override val callId: String,
        /**
         * Required. ID to let user identify remote echo of their own events
         */
        @Json(name = "party_id") override val partyId: String? = null,
        /**
         * An identifier for the call replacement itself, generated by the transferor.
         */
        @Json(name = "replacement_id") val replacementId: String? = null,
        /**
         * Optional. If specified, the transferee client waits for an invite to this room and joins it
         * (possibly waiting for user confirmation) and then continues the transfer in this room.
         * If absent, the transferee contacts the Matrix User ID given in the target_user field in a room of its choosing.
         */
        @Json(name = "target_room") val targetRoomId: String? = null,
        /**
         * An object giving information about the transfer target
         */
        @Json(name = "target_user") val targetUser: TargetUser? = null,
        /**
         * If specified, gives the call ID for the transferee's client to use when placing the replacement call.
         * Mutually exclusive with await_call
         */
        @Json(name = "create_call") val createCall: String? = null,
        /**
         * If specified, gives the call ID that the transferee's client should wait for.
         * Mutually exclusive with create_call.
         */
        @Json(name = "await_call") val awaitCall: String? = null,
        /**
         * Required. The version of the VoIP specification this messages adheres to.
         */
        @Json(name = "version") override val version: String?
) : CallSignalingContent  {

    @JsonClass(generateAdapter = true)
    data class TargetUser(
            /**
             * Required. The matrix user ID of the transfer target
             */
            @Json(name = "id") val id: String,
            /**
             * Optional. The display name of the transfer target.
             */
            @Json(name = "display_name") val displayName: String?,
            /**
             * Optional. The avatar URL of the transfer target.
             */
            @Json(name = "avatar_url") val avatarUrl: String?
    )
}
